﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Domain.Helpers;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Models.Patient;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Utilities;

    /// <summary>
    /// The patient emergency contacts.
    /// </summary>
    [Authorize]
    [Route("api/patient-emergency-contacts")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class PatientEmergencyContactsController : BaseController
    {
        /// <summary>
        /// The patient emergency services.
        /// </summary>
        private readonly IPatientEmergencyService patientEmergencyServices;

        /// <summary>
        /// The AES helper.
        /// </summary>
        private readonly IAESHelper aesHelper;

        /// <inheritdoc />
        public PatientEmergencyContactsController(IPatientEmergencyService patientEmergencyServices, IAESHelper aesHelper)
        {
            this.patientEmergencyServices = patientEmergencyServices;
            this.aesHelper = aesHelper;
        }

        /// <summary>
        /// The fetch emergency contacts async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<PatientEmergencyModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody]FindPatientRequest model)
        {
            model = (FindPatientRequest)EmptyFilter.Handler(model);
            var patientId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedPatientId));
            var patientEmergencyContacts = await this.patientEmergencyServices.FetchAsync(patientId);

            return this.Success(patientEmergencyContacts);
        }

        /// <summary>
        /// The add emergency contacts async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody]PatientEmergencyModel model)
        {
            var response = await this.patientEmergencyServices.AddAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Emergency contact has already been exists with us.");
                case 0:
                    return this.ServerError();
                default:
                    return this.Success("Emergency contact details updated successfully.");
            }
        }

        /// <summary>
        /// The update emergency contacts async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody]PatientEmergencyModel model)
        {
            var response = await this.patientEmergencyServices.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Emergency contact has already been exists with us.");
                case 0:
                    return this.ServerError();
                default:
                    return this.Success("Emergency contact details updated successfully.");
            }
        }

        /// <summary>
        /// The delete emergency contacts async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody]PatientEmergencyModel model)
        {
            var response = await this.patientEmergencyServices.DeleteAsync(model.PatientEmergencyId);
            if (response != 0)
            {
                return this.Success("Emergency contact details deleted successfully.");
            }

            return this.ServerError();
        }
    }
}